<?php
require_once 'security_headers.php';
session_start();
require 'db.php';

// Fetch leaderboard data
$leaderboard = [];
try {
    // Explicitly select only needed columns to prevent data leakage
    $stmt = $pdo->prepare("SELECT rank, username, display_name, score, round_reached, achieved_at FROM leaderboard ORDER BY rank ASC LIMIT :limit");
    $stmt->bindValue(':limit', 10, PDO::PARAM_INT);
    $stmt->execute();
    $leaderboard = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    // Silently fail if view doesn't exist
    error_log("Leaderboard fetch failed: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html>
<head>
    <title>Asteroids Game</title>
    <link rel="stylesheet" type="text/css" href="style.css">
    <style>
        .landing-container {
            text-align: center;
            padding: 50px;
            background: linear-gradient(135deg, #0f0f23 0%, #1a1a2e 100%);
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
        }
        
        .game-title {
            font-size: 4em;
            color: #4CAF50;
            margin-bottom: 20px;
            text-shadow: 0 0 10px rgba(76, 175, 80, 0.5);
        }
        
        .game-description {
            font-size: 1.2em;
            color: #ccc;
            margin-bottom: 40px;
            max-width: 600px;
        }
        
        .auth-buttons {
            display: flex;
            gap: 20px;
            margin-top: 30px;
        }
        
        .btn {
            padding: 15px 30px;
            font-size: 1.1em;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .btn-primary {
            background: #4CAF50;
            color: white;
        }
        
        .btn-secondary {
            background: transparent;
            color: #4CAF50;
            border: 2px solid #4CAF50;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(76, 175, 80, 0.3);
        }
        
        .features {
            display: flex;
            justify-content: center;
            gap: 30px;
            margin: 50px 0;
            flex-wrap: wrap;
        }
        
        .feature {
            background: rgba(255, 255, 255, 0.1);
            padding: 20px;
            border-radius: 8px;
            width: 200px;
        }
        
        .feature h3 {
            color: #4CAF50;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="landing-container">
        <h1 class="game-title">🚀 ASTEROIDS</h1>
        
        <p class="game-description">
            Blast your way through waves of asteroids in this classic arcade game! 
            Survive as long as you can, earn high scores, and compete with players worldwide.
        </p>
        
        <?php if (isset($_SESSION['user_id'])): ?>
            <div class="auth-buttons">
                <a href="game.php" class="btn btn-primary">Play Game</a>
                <a href="logout.php" class="btn btn-secondary">Logout</a>
            </div>
        <?php else: ?>
            <div class="auth-buttons">
                <a href="login.php" class="btn btn-primary">Login</a>
                <a href="register.php" class="btn btn-secondary">Register</a>
            </div>
        <?php endif; ?>
        
        <div class="features">
            <div class="feature">
                <h3>🎯 Progressive Difficulty</h3>
                <p>Game gets harder each round with more asteroids</p>
            </div>
            <div class="feature">
                <h3>🏆 Score Tracking</h3>
                <p>Track your scores and compete for the top spot</p>
            </div>
            <div class="feature">
                <h3>🎮 Multiplayer</h3>
                <p>Play against other players and climb the leaderboard</p>
            </div>
            <div class="feature">
                <h3>💾 Save Progress</h3>
                <p>Your games and scores are saved automatically</p>
            </div>
        </div>
        
        <div class="game-preview">
            <h2>Preview Game</h2>
            <canvas id="previewCanvas" width="400" height="300" style="border: 2px solid #333; margin-top: 20px;"></canvas>
        </div>
        
        <?php if (!empty($leaderboard)): ?>
        <div class="leaderboard-section">
            <h2 style="color: #4CAF50; margin-bottom: 20px;">🏆 Top Players Leaderboard</h2>
            <table class="leaderboard-table">
                <thead>
                    <tr>
                        <th>Rank</th>
                        <th>Player</th>
                        <th>Score</th>
                        <th>Round</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($leaderboard as $entry): ?>
                    <tr>
                        <td class="rank-<?php echo $entry['rank']; ?>">#<?php echo $entry['rank']; ?></td>
                        <td><?php echo htmlspecialchars($entry['display_name'] ?: $entry['username']); ?></td>
                        <td><?php echo number_format($entry['score']); ?></td>
                        <td><?php echo $entry['round_reached']; ?></td>
                        <td><?php echo date('M d, Y', strtotime($entry['achieved_at'])); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>

    <script>
        // Simple preview animation
        const canvas = document.getElementById('previewCanvas');
        const ctx = canvas.getContext('2d');
        
        function drawPreview() {
            ctx.fillStyle = 'black';
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            // Draw some sample coins and spaceship
            ctx.strokeStyle = 'white';
            ctx.fillStyle = 'white';
            
            // Spaceship (more detailed design)
            ctx.save();
            ctx.translate(canvas.width/2, canvas.height/2);
            
            // Main body
            ctx.beginPath();
            ctx.moveTo(20, 0);
            ctx.lineTo(-15, -12);
            ctx.lineTo(-10, -8);
            ctx.lineTo(-10, 8);
            ctx.lineTo(-15, 12);
            ctx.closePath();
            ctx.stroke();
            
            // Cockpit
            ctx.beginPath();
            ctx.arc(5, 0, 4, 0, Math.PI * 2);
            ctx.stroke();
            
            // Engine flames
            ctx.strokeStyle = 'orange';
            ctx.beginPath();
            ctx.moveTo(-10, -5);
            ctx.lineTo(-20, -8);
            ctx.moveTo(-10, 5);
            ctx.lineTo(-20, 8);
            ctx.stroke();
            
            ctx.restore();
            
            // Coins instead of asteroids
            function drawCoin(x, y, radius) {
                // Outer circle
                ctx.beginPath();
                ctx.arc(x, y, radius, 0, Math.PI * 2);
                ctx.stroke();
                
                // Inner circle (coin design)
                ctx.beginPath();
                ctx.arc(x, y, radius * 0.7, 0, Math.PI * 2);
                ctx.stroke();
                
                // Dollar sign in center
                ctx.font = `${radius}px Arial`;
                ctx.textAlign = 'center';
                ctx.textBaseline = 'middle';
                ctx.fillText('$', x, y);
            }
            
            drawCoin(100, 100, 20);
            drawCoin(300, 150, 15);
            drawCoin(200, 250, 10);
        }
        
        drawPreview();
    </script>
</body>
</html>