# Production Deployment Checklist for Asteroids Website

## ✅ Completed Security Improvements

### 1. **Database Security**

- ✅ Removed hardcoded credentials from `db.php`
- ✅ Created environment configuration system (`config.php` and `config.env`)
- ✅ Added proper error handling that doesn't expose database details

### 2. **Code Security**

- ✅ Removed all console.log and error_log statements from production code
- ✅ Added CSRF protection to all forms
- ✅ Added input sanitization for user inputs
- ✅ Implemented rate limiting for authentication endpoints

### 3. **File Security**

- ✅ Created `.gitignore` to exclude sensitive files
- ✅ Protected sensitive files via `.htaccess`
- ✅ SQL dump files are now gitignored

### 4. **Session Security**

- ✅ Session regeneration on login
- ✅ Secure session cookie settings
- ✅ HttpOnly and SameSite cookie attributes

### 5. **Headers Security**

- ✅ Content Security Policy configured
- ✅ X-Frame-Options, X-Content-Type-Options, X-XSS-Protection headers
- ✅ Referrer Policy and Permissions Policy set

## 📋 Required Actions Before Deployment

### 1. **Environment Configuration**

1. Copy `config.env.example` to `config.env`
2. Update the following values in `config.env`:

   ```env
   DB_HOST=your_production_host
   DB_NAME=your_database_name
   DB_USER=your_database_user
   DB_PASS=your_secure_password
   ENVIRONMENT=production
   ```

### 2. **Database Setup**

1. Import the database schema (use the SQL files locally, don't upload them)
2. Create the rate_limit table (will be auto-created by rate_limiter.php)
3. Ensure proper database user permissions (only necessary privileges)

### 3. **SSL Certificate**

1. Install SSL certificate on your server
2. After SSL is installed, uncomment HTTPS redirect in `.htaccess`:

   ```apache
   RewriteCond %{HTTPS} !=on
   RewriteRule ^(.*)$ https://%{HTTP_HOST}/$1 [R=301,L]
   ```

### 4. **File Cleanup**

1. Delete these files from production server:
   - `markxwyo_Asteroids.sql`
   - `alter_to_utf8mb4.sql`
   - `error_log`
   - `example_with_ads.php` (if not needed)
   - `config.env.example` (after creating config.env)

### 5. **Permissions**

1. Set proper file permissions:

   ```bash
   chmod 644 *.php
   chmod 644 *.js
   chmod 644 *.css
   chmod 644 .htaccess
   chmod 600 config.env
   ```

### 6. **AdSense Configuration** (Optional)

1. Update `ADSENSE_PUBLISHER_ID` in `config.env` with your actual AdSense ID
2. Implement ad units where needed using the `adsense_config.php` functions

## 🔒 Security Best Practices

1. **Regular Updates**
   - Keep PHP version updated (minimum PHP 7.4 recommended)
   - Update dependencies regularly
   - Monitor security advisories

2. **Monitoring**
   - Set up error logging to files (not displayed to users)
   - Monitor failed login attempts
   - Regular security audits

3. **Backups**
   - Regular database backups
   - File system backups
   - Test restore procedures

4. **Additional Recommendations**
   - Consider implementing 2FA for admin accounts
   - Use a Web Application Firewall (WAF)
   - Implement proper logging and monitoring
   - Regular security scans

## 🔐 Critical Security Recommendations (Senior Dev Audit)

### 1. **Move Sensitive Files Outside Web Root**

- **Priority: HIGH**
- Move `db.php` and `config.php` above document root
- Use PHP include path configuration
- Example structure:

  ```plaintext
     /home/user/
       ├── includes/          (outside web root)
       │   ├── db.php
       │   └── config.php
       └── public_html/       (web root)
           └── [website files]
  ```

### 2. **Implement Strong Password Policy**

- **Priority: MEDIUM**
- Add to `register.php` validation:

  ```php
   // Require 8+ chars with uppercase, lowercase, and number
   if (!preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).{8,}$/', $_POST['password'])) {
       $error = "Password must be at least 8 characters with uppercase, lowercase, and number";
   }
  ```

### 3. **Add Session Timeout**

- **Priority: MEDIUM**
- Add to `security_headers.php`:

  ```php
   // Set session timeout to 1 hour
   ini_set('session.gc_maxlifetime', 3600);
   
   // Check for session timeout
   if (isset($_SESSION['LAST_ACTIVITY']) && (time() - $_SESSION['LAST_ACTIVITY'] > 3600)) {
       session_unset();
       session_destroy();
       header("Location: login.php?timeout=1");
   }
   $_SESSION['LAST_ACTIVITY'] = time();
  ```

### 4. **Implement CSP Nonce for Inline Scripts**

- **Priority: LOW**
- Generate nonce for each request:

  ```php
   $nonce = base64_encode(random_bytes(16));
   header("Content-Security-Policy: script-src 'self' 'nonce-$nonce'");
  ```

- Add to inline scripts: `<script nonce="<?php echo $nonce; ?>">`

### 5. **Add Database Query Logging**

- **Priority: LOW**
- For development/debugging:

  ```php
   // Add to db.php for development only
   if (getConfig('ENVIRONMENT') !== 'production') {
       $pdo->setAttribute(PDO::ATTR_STATEMENT_CLASS, 
           ['LoggedPDOStatement', [$pdo]]);
   }
  ```

- Monitor for suspicious patterns
- Log slow queries for optimization

## �� Deployment Steps

1. **Pre-deployment**
   - [ ] Create config.env with production values
   - [ ] Test locally with production-like settings
   - [ ] Backup current production (if updating)

2. **Deployment**
   - [ ] Upload files (excluding sensitive ones)
   - [ ] Set proper file permissions
   - [ ] Import database schema
   - [ ] Test database connection

3. **Post-deployment**
   - [ ] Verify HTTPS is working
   - [ ] Test login/registration with rate limiting
   - [ ] Verify game functionality
   - [ ] Check error handling
   - [ ] Monitor for any issues

## ⚠️ Important Notes

- Never commit `config.env` to version control
- Always use HTTPS in production
- Keep the `.gitignore` file updated
- Regular security audits are recommended
- Consider using a CDN for static assets
- Implement proper logging (to files, not screen)

## 📞 Support

If you encounter issues:

1. Check PHP error logs (not displayed to users)
2. Verify database connectivity
3. Ensure all required PHP extensions are installed
4. Check file permissions

---

Last Updated: November 2024
Version: 1.0
