export class Ship {
    constructor(x, y, scaleFactor = 1) {
        this.x = x;
        this.y = y;
        this.angle = 0;
        this.vx = 0;
        this.vy = 0;
        this.rotateLeft = false;
        this.rotateRight = false;
        this.thrusting = false;
        this.radius = 24 * scaleFactor;
        this.thrustIntensity = 0;
        this.visible = true;
        
        this.shieldActive = false;
        this.shieldEnergy = 100;
        this.shieldRechargeRate = 0.5;
        this.shieldDrainRate = 2;
        this.shieldRadius = this.radius * 1.5;
        
        this.primaryColor = '#ffffff';
        this.secondaryColor = '#6666ff';
        this.accentColor = '#00ffff';
        this.cockpitColor = '#66ccff';
        
        this.engineHeat = 0;
        this.engineGlow = 0;
    }
    
    update(canvasWidth, canvasHeight) {
        if (this.rotateLeft) this.angle -= 0.05;
        if (this.rotateRight) this.angle += 0.05;
        
        if (this.thrusting) {
            this.vx += Math.cos(this.angle) * 0.1;
            this.vy += Math.sin(this.angle) * 0.1;
            this.thrustIntensity = Math.min(this.thrustIntensity + 0.1, 1);
            this.engineHeat = Math.min(this.engineHeat + 2, 100);
            this.engineGlow = Math.min(this.engineGlow + 5, 255);
        } else {
            this.thrustIntensity = Math.max(this.thrustIntensity - 0.05, 0);
            this.engineHeat = Math.max(this.engineHeat - 1, 0);
            this.engineGlow = Math.max(this.engineGlow - 2, 0);
        }
        
        if (this.shieldActive && this.shieldEnergy > 0) {
            this.shieldEnergy -= this.shieldDrainRate;
            if (this.shieldEnergy <= 0) {
                this.shieldActive = false;
                this.shieldEnergy = 0;
            }
        } else if (!this.shieldActive && this.shieldEnergy < 100) {
            this.shieldEnergy += this.shieldRechargeRate;
        }
        
        this.x += this.vx;
        this.y += this.vy;
        
        if (this.x < 0) this.x = canvasWidth;
        if (this.x > canvasWidth) this.x = 0;
        if (this.y < 0) this.y = canvasHeight;
        if (this.y > canvasHeight) this.y = 0;
        
        this.vx *= 0.99;
        this.vy *= 0.99;
    }
    
    reset(canvasWidth, canvasHeight) {
        this.x = canvasWidth / 2;
        this.y = canvasHeight / 2;
        this.vx = 0;
        this.vy = 0;
        this.angle = 0;
        this.thrustIntensity = 0;
        this.engineHeat = 0;
        this.engineGlow = 0;
        this.shieldEnergy = 100;
        this.shieldActive = false;
    }
    
    draw(ctx) {
        if (this.visible === false) return;
        
        if (this.thrusting || this.thrustIntensity > 0) {
            this.drawThrustFlames(ctx);
        }
        
        ctx.save();
        ctx.translate(this.x, this.y);
        ctx.rotate(this.angle);
        
        if (this.shieldActive && this.shieldEnergy > 0) {
            ctx.shadowColor = this.accentColor;
            ctx.shadowBlur = 20;
        } else {
            const glowIntensity = Math.max(this.engineGlow / 255, 0.3);
            ctx.shadowColor = `rgba(255, 100, 0, ${glowIntensity})`;
            ctx.shadowBlur = 10 + this.engineGlow / 25;
        }
        
        const gradientX1 = -20 * Math.cos(this.angle);
        const gradientY1 = -20 * Math.sin(this.angle);
        const gradient = ctx.createLinearGradient(gradientX1, gradientY1, 30, 0);
        gradient.addColorStop(0, this.secondaryColor);
        gradient.addColorStop(0.5, this.primaryColor);
        gradient.addColorStop(1, this.secondaryColor);
        
        ctx.fillStyle = gradient;
        ctx.strokeStyle = 'white';
        ctx.lineWidth = 3;
        ctx.lineJoin = 'round';
        
        ctx.beginPath();
        ctx.moveTo(30, 0);
        ctx.lineTo(-20, -20);
        ctx.lineTo(-16, -10);
        ctx.lineTo(-30, -10);
        ctx.lineTo(-24, 0);
        ctx.lineTo(-30, 10);
        ctx.lineTo(-16, 10);
        ctx.lineTo(-20, 20);
        ctx.closePath();
        ctx.fill();
        ctx.stroke();
        
        const heatColor = this.getEngineHeatColor();
        ctx.fillStyle = heatColor;
        ctx.fillRect(-24, -6, 8, 12);
        
        ctx.fillStyle = '#222222';
        ctx.fillRect(-22, -4, 4, 8);
        
        this.drawCockpit(ctx);
        this.drawShipDetails(ctx);
        
        ctx.restore();
        
        if (this.shieldActive && this.shieldEnergy > 0) {
            this.drawShield(ctx);
        }
    }
    
    drawCockpit(ctx) {
        const glassGradient = ctx.createRadialGradient(5, 0, 0, 5, 0, 8);
        glassGradient.addColorStop(0, '#aaddff');
        glassGradient.addColorStop(0.7, this.cockpitColor);
        glassGradient.addColorStop(1, '#3388aa');
        
        ctx.fillStyle = glassGradient;
        ctx.beginPath();
        ctx.arc(10, 0, 6, 0, Math.PI * 2);
        ctx.fill();
        
        ctx.strokeStyle = '#333333';
        ctx.lineWidth = 1;
        ctx.beginPath();
        ctx.arc(10, 0, 6, 0, Math.PI * 2);
        ctx.stroke();
        
        ctx.fillStyle = 'rgba(255, 255, 255, 0.3)';
        ctx.beginPath();
        ctx.arc(8, -2, 2, 0, Math.PI * 2);
        ctx.fill();
    }
    
    drawShipDetails(ctx) {
        ctx.strokeStyle = 'rgba(100, 100, 255, 0.5)';
        ctx.lineWidth = 1;
        
        ctx.beginPath();
        ctx.moveTo(25, 0);
        ctx.lineTo(-15, 0);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.moveTo(-10, -8);
        ctx.lineTo(-25, -8);
        ctx.moveTo(-10, 8);
        ctx.lineTo(-25, 8);
        ctx.stroke();
        
        ctx.fillStyle = this.accentColor;
        ctx.beginPath();
        ctx.arc(28, 0, 2, 0, Math.PI * 2);
        ctx.fill();
        
        ctx.beginPath();
        ctx.arc(-28, -8, 1.5, 0, Math.PI * 2);
        ctx.fill();
        ctx.beginPath();
        ctx.arc(-28, 8, 1.5, 0, Math.PI * 2);
        ctx.fill();
    }
    
    drawThrustFlames(ctx) {
        ctx.save();
        ctx.translate(this.x, this.y);
        ctx.rotate(this.angle);
        
        const baseLength = 40 * this.thrustIntensity;
        const baseWidth = 16 * this.thrustIntensity;
        
        const heat = this.engineHeat / 100;
        const innerColor = this.getFlameColor(heat, 0);
        const middleColor = this.getFlameColor(heat, 0.5);
        const outerColor = this.getFlameColor(heat, 1);
        
        const layers = [
            {color: innerColor, width: baseWidth * 0.8, length: baseLength * 0.7},
            {color: middleColor, width: baseWidth * 0.6, length: baseLength * 0.9},
            {color: outerColor, width: baseWidth * 0.4, length: baseLength * 1.1}
        ];
        
        layers.forEach(layer => {
            ctx.strokeStyle = layer.color;
            ctx.lineWidth = layer.width;
            ctx.lineCap = 'round';
            ctx.shadowColor = layer.color;
            ctx.shadowBlur = layer.width * 2;
            
            ctx.beginPath();
            ctx.moveTo(-24, 0);
            ctx.lineTo(-24 - layer.length * (0.7 + Math.random() * 0.3), 
                       (Math.random() - 0.5) * 4);
            ctx.stroke();
            
            ctx.beginPath();
            ctx.moveTo(-24, -4);
            ctx.lineTo(-24 - layer.length * (0.5 + Math.random() * 0.2), 
                       -8 + (Math.random() - 0.5) * 2);
            ctx.stroke();
            
            ctx.beginPath();
            ctx.moveTo(-24, 4);
            ctx.lineTo(-24 - layer.length * (0.5 + Math.random() * 0.2), 
                       8 + (Math.random() - 0.5) * 2);
            ctx.stroke();
        });
        
        if (this.thrustIntensity > 0.3) {
            const particleCount = Math.floor(this.thrustIntensity * 8);
            for (let i = 0; i < particleCount; i++) {
                const offsetX = -24 - Math.random() * baseLength * 1.2;
                const offsetY = (Math.random() - 0.5) * 12;
                const size = Math.random() * 4 * this.thrustIntensity;
                const alpha = 0.3 + Math.random() * 0.7;
                
                const particleColors = ['#FFFF00', '#FF6600', '#FF3300', '#FF9900'];
                ctx.fillStyle = particleColors[Math.floor(Math.random() * particleColors.length)];
                ctx.globalAlpha = alpha;
                
                ctx.beginPath();
                ctx.arc(offsetX, offsetY, size, 0, Math.PI * 2);
                ctx.fill();
                
                ctx.globalAlpha = 1;
            }
        }
        
        ctx.restore();
    }
    
    drawShield(ctx) {
        ctx.save();
        ctx.translate(this.x, this.y);
        
        const pulse = Math.sin(Date.now() * 0.01) * 0.1 + 0.9;
        const alpha = (this.shieldEnergy / 100) * 0.7;
        
        const shieldGradient = ctx.createRadialGradient(0, 0, this.radius, 0, 0, this.shieldRadius);
        shieldGradient.addColorStop(0, `rgba(0, 200, 255, ${alpha * 0.3})`);
        shieldGradient.addColorStop(1, `rgba(0, 100, 255, ${alpha * 0.1})`);
        
        ctx.strokeStyle = shieldGradient;
        ctx.fillStyle = shieldGradient;
        ctx.lineWidth = 2;
        ctx.setLineDash([8, 4]);
        
        ctx.beginPath();
        ctx.arc(0, 0, this.shieldRadius * pulse, 0, Math.PI * 2);
        ctx.stroke();
        
        ctx.beginPath();
        ctx.arc(0, 0, this.shieldRadius * pulse * 0.9, 0, Math.PI * 2);
        ctx.stroke();
        
        ctx.setLineDash([]);
        ctx.restore();
    }
    
    getEngineHeatColor() {
        const heat = this.engineHeat / 100;
        if (heat < 0.3) return '#333333';
        if (heat < 0.6) return '#664422';
        if (heat < 0.8) return '#aa5522';
        return '#ff6622';
    }
    
    getFlameColor(heat, position) {
        const r = Math.floor(255 * (1 - position * 0.5) * (0.8 + heat * 0.2));
        const g = Math.floor(150 * (1 - position) * (0.6 + heat * 0.4));
        const b = Math.floor(50 * (1 - position));
        return `rgb(${r}, ${g}, ${b})`;
    }
    
    toggleShield() {
        if (this.shieldEnergy > 10) {
            this.shieldActive = !this.shieldActive;
        } else {
            this.shieldActive = false;
        }
    }
    
    setColors(primary, secondary, accent, cockpit) {
        this.primaryColor = primary || this.primaryColor;
        this.secondaryColor = secondary || this.secondaryColor;
        this.accentColor = accent || this.accentColor;
        this.cockpitColor = cockpit || this.cockpitColor;
    }
    
    setColorScheme(scheme) {
        const schemes = {
            default: {
                primary: '#ffffff',
                secondary: '#6666ff', 
                accent: '#00ffff',
                cockpit: '#66ccff'
            },
            red: {
                primary: '#ff8888',
                secondary: '#ff3333',
                accent: '#ff0000',
                cockpit: '#ff6666'
            },
            green: {
                primary: '#88ff88',
                secondary: '#33cc33',
                accent: '#00ff00',
                cockpit: '#66ff66'
            },
            stealth: {
                primary: '#888888',
                secondary: '#444444',
                accent: '#222222',
                cockpit: '#666666'
            },
            gold: {
                primary: '#ffdd88',
                secondary: '#ffaa00',
                accent: '#ff8800',
                cockpit: '#ffcc66'
            }
        };
        
        if (schemes[scheme]) {
            this.setColors(
                schemes[scheme].primary,
                schemes[scheme].secondary,
                schemes[scheme].accent,
                schemes[scheme].cockpit
            );
        }
    }
}
