export class Explosion {
    constructor(x, y) {
        this.x = x;
        this.y = y;
        this.particles = [];
        this.shockwaves = [];
        this.finished = false;
        this.age = 0;
        this.maxAge = 60;
        
        this.createParticles();
        this.createShockwaves();
    }
    
    createParticles() {
        const particleCount = 20 + Math.random() * 15;
        
        for (let i = 0; i < particleCount; i++) {
            const angle = (Math.PI * 2 * i) / particleCount + Math.random() * 0.5;
            const speed = 2 + Math.random() * 4;
            const size = 1 + Math.random() * 3;
            
            const particleType = Math.random();
            let color;
            
            if (particleType < 0.3) {
                color = '#ffffff';
            } else if (particleType < 0.6) {
                color = `hsl(${30 + Math.random() * 30}, 100%, ${50 + Math.random() * 30}%)`;
            } else {
                color = `hsl(${0 + Math.random() * 30}, 100%, ${40 + Math.random() * 30}%)`;
            }
            
            this.particles.push({
                x: this.x,
                y: this.y,
                vx: Math.cos(angle) * speed,
                vy: Math.sin(angle) * speed,
                size: size,
                color: color,
                life: 1,
                decay: 0.015 + Math.random() * 0.02,
                trail: []
            });
        }
    }
    
    createShockwaves() {
        const waveCount = 2 + Math.floor(Math.random() * 2);
        
        for (let i = 0; i < waveCount; i++) {
            this.shockwaves.push({
                radius: 5 + i * 10,
                maxRadius: 30 + i * 20,
                speed: 2 + i * 0.5,
                opacity: 0.8 - i * 0.2,
                color: i === 0 ? '#ffffff' : '#ffaa00'
            });
        }
    }
    
    update() {
        this.age++;
        
        for (let i = this.particles.length - 1; i >= 0; i--) {
            const particle = this.particles[i];
            
            particle.trail.push({x: particle.x, y: particle.y, life: 1});
            if (particle.trail.length > 5) {
                particle.trail.shift();
            }
            
            particle.trail.forEach(point => {
                point.life -= 0.2;
            });
            
            particle.x += particle.vx;
            particle.y += particle.vy;
            
            particle.vx *= 0.98;
            particle.vy *= 0.98;
            
            particle.vy += 0.05;
            
            particle.life -= particle.decay;
            
            particle.size *= 0.98;
            
            if (particle.life <= 0 || particle.size < 0.1) {
                this.particles.splice(i, 1);
            }
        }
        
        for (let i = this.shockwaves.length - 1; i >= 0; i--) {
            const wave = this.shockwaves[i];
            wave.radius += wave.speed;
            wave.opacity -= 0.02;
            
            if (wave.opacity <= 0 || wave.radius > wave.maxRadius) {
                this.shockwaves.splice(i, 1);
            }
        }
        
        if (this.age > this.maxAge && this.particles.length === 0 && this.shockwaves.length === 0) {
            this.finished = true;
        }
    }
    
    draw(ctx) {
        ctx.save();
        
        this.shockwaves.forEach(wave => {
            ctx.globalAlpha = wave.opacity;
            ctx.strokeStyle = wave.color;
            ctx.lineWidth = 2;
            ctx.shadowColor = wave.color;
            ctx.shadowBlur = 10;
            
            ctx.beginPath();
            ctx.arc(this.x, this.y, wave.radius, 0, Math.PI * 2);
            ctx.stroke();
        });
        
        this.particles.forEach(particle => {
            particle.trail.forEach((point, index) => {
                if (point.life > 0) {
                    ctx.globalAlpha = point.life * particle.life * 0.3;
                    ctx.fillStyle = particle.color;
                    ctx.shadowColor = particle.color;
                    ctx.shadowBlur = 5;
                    
                    const trailSize = particle.size * point.life * 0.5;
                    ctx.beginPath();
                    ctx.arc(point.x, point.y, trailSize, 0, Math.PI * 2);
                    ctx.fill();
                }
            });
            
            ctx.globalAlpha = particle.life;
            ctx.fillStyle = particle.color;
            ctx.shadowColor = particle.color;
            ctx.shadowBlur = 10;
            
            ctx.beginPath();
            ctx.arc(particle.x, particle.y, particle.size, 0, Math.PI * 2);
            ctx.fill();
            
            if (particle.color === '#ffffff' || particle.life > 0.7) {
                ctx.globalAlpha = particle.life * 0.5;
                ctx.beginPath();
                ctx.arc(particle.x, particle.y, particle.size * 2, 0, Math.PI * 2);
                ctx.fill();
            }
        });
        
        ctx.restore();
    }
}
