export class Bullet {
    constructor(x, y, angle, scaleFactor = 1) {
        this.x = x;
        this.y = y;
        this.angle = angle;
        this.speed = 8 * scaleFactor;
        this.vx = Math.cos(angle) * this.speed;
        this.vy = Math.sin(angle) * this.speed;
        this.radius = 4 * scaleFactor;
        this.length = 20 * scaleFactor;
        this.trail = [];
        this.maxTrailLength = 8;
        this.lifetime = 60;
        this.age = 0;
        
        this.coreColor = '#ffffff';
        this.glowColor = '#00ffff';
        this.trailColor = '#ffaa00';
        this.energyPulse = 0;
    }
    
    update() {
        this.trail.push({x: this.x, y: this.y, age: 0});
        if (this.trail.length > this.maxTrailLength) {
            this.trail.shift();
        }
        
        this.trail.forEach(point => point.age++);
        
        this.x += this.vx;
        this.y += this.vy;
        
        this.age++;
        
        this.energyPulse = Math.sin(this.age * 0.3) * 0.5 + 0.5;
    }
    
    draw(ctx) {
        this.drawTrail(ctx);
        
        ctx.save();
        ctx.translate(this.x, this.y);
        ctx.rotate(this.angle);
        
        const glowIntensity = 0.7 + this.energyPulse * 0.3;
        ctx.shadowColor = this.glowColor;
        ctx.shadowBlur = 15 * glowIntensity;
        
        const coreGradient = ctx.createRadialGradient(0, 0, 0, 0, 0, this.radius * 2);
        coreGradient.addColorStop(0, this.coreColor);
        coreGradient.addColorStop(0.3, this.glowColor);
        coreGradient.addColorStop(1, 'rgba(0, 255, 255, 0)');
        
        ctx.fillStyle = coreGradient;
        ctx.beginPath();
        ctx.arc(0, 0, this.radius * 2, 0, Math.PI * 2);
        ctx.fill();
        
        const bulletGradient = ctx.createLinearGradient(-this.length/2, 0, this.length/2, 0);
        bulletGradient.addColorStop(0, 'rgba(255, 255, 0, 0.3)');
        bulletGradient.addColorStop(0.5, 'rgba(255, 255, 255, 1)');
        bulletGradient.addColorStop(1, 'rgba(255, 255, 0, 0.3)');
        
        ctx.fillStyle = bulletGradient;
        ctx.strokeStyle = this.coreColor;
        ctx.lineWidth = 2;
        
        ctx.beginPath();
        ctx.moveTo(-this.length/2, -this.radius/2);
        ctx.lineTo(this.length/2, 0);
        ctx.lineTo(-this.length/2, this.radius/2);
        ctx.closePath();
        ctx.fill();
        ctx.stroke();
        
        ctx.strokeStyle = this.glowColor;
        ctx.lineWidth = 1;
        ctx.globalAlpha = 0.6 * glowIntensity;
        
        for (let i = 0; i < 2; i++) {
            const ringRadius = this.radius * (1.5 + i * 0.5) * (1 + this.energyPulse * 0.3);
            ctx.beginPath();
            ctx.arc(0, 0, ringRadius, 0, Math.PI * 2);
            ctx.stroke();
        }
        
        ctx.restore();
    }
    
    drawTrail(ctx) {
        ctx.save();
        
        this.trail.forEach((point, index) => {
            const alpha = (1 - point.age / this.maxTrailLength) * 0.6;
            const size = this.radius * (1 - point.age / this.maxTrailLength) * 1.5;
            
            ctx.globalAlpha = alpha;
            ctx.fillStyle = this.trailColor;
            ctx.shadowColor = this.trailColor;
            ctx.shadowBlur = 5;
            
            ctx.beginPath();
            ctx.arc(point.x, point.y, size, 0, Math.PI * 2);
            ctx.fill();
        });
        
        ctx.restore();
    }
    
    isExpired() {
        return this.age > this.lifetime;
    }
}
